# Bouncing ball game with spikes
# Inspired by the iOS game "Don't Touch The Spikes" by Ketchapp

import sys, pygame, time, os, random, platform
sys.dont_write_bytecode = True

pygame.init()
pygame.mixer.init()

WIDTH, HEIGHT = 480, 640
W_SPIKE, H_SPIKE = 12, 45
RADIUS = 18
FLOOR_SIZE = 21
GRAVITY = 0.45
HSPEED, HSPEED_DELTA = 3.5, 0.15
JUMP_VELOCITY = -9
FPS = 60

COLORS = {
    'ball': (40, 120, 220),
    'ball_border': (20, 70, 160),
    'line': (200, 200, 200),
    'spike': (180, 80, 80)
}

# There's probably a better way to do this, if it works it works
BACKGROUNDS = [
    (255, 255, 255),
    (255, 235, 235),
    (255, 214, 214),
    (255, 194, 194),
    (255, 173, 173),
    (255, 153, 153),
    (255, 133, 133),
    (255, 112, 112),
    (255, 92, 92),
    (255, 71, 71),
    (243, 51, 51),
    (230, 46, 46),
    (204, 41, 41),
    (179, 36, 36),
    (153, 31, 31),
    (128, 26, 26),
    (102, 20, 20),
    (77, 15, 15),
    (51, 10, 10),
    (26, 5, 5),
    (0, 0, 0) 
]

SOUNDS = {
    'jump': pygame.mixer.Sound('jump.wav'),
    'bounce': pygame.mixer.Sound('bounce.wav'),
    'crash': pygame.mixer.Sound('crash.wav'),
    'levelup': pygame.mixer.Sound('levelup.wav')
}

def reset_state():
    return [
        WIDTH // 2, HEIGHT // 2,    # Initial x and y pos
        0.0,                        # Initial vertical velocity
        True,                       # Waiting for first jump
        0.0                         # Initial horizontal velocity
    ]

def circle_rect_collides(cx, cy, radius, rect):
    closest_x, closest_y = max(rect.left, min(cx, rect.right)), max(rect.top, min(cy, rect.bottom))
    dx, dy = cx - closest_x, cy - closest_y
    return dx * dx + dy * dy <= radius * radius

def spawn_spike(side, existing_spikes):
    max_attempts = 7
    for _ in range(max_attempts):
        y_center = random.randint(
            FLOOR_SIZE + H_SPIKE // 2 + RADIUS, 
            HEIGHT - FLOOR_SIZE - H_SPIKE // 2 - RADIUS
        )
        top = y_center - H_SPIKE // 2
        
        valid = True
        for spike in existing_spikes:
            # Minimum distance between spikes before they look cramped
            # This is so the game doesn't become RNG-dependent
            if abs(spike.centery - y_center) < (HEIGHT // 10):
                valid = False
                break
        
        if valid:
            if side == 'left':
                return pygame.Rect(0, top, W_SPIKE, H_SPIKE)
            return pygame.Rect(WIDTH - W_SPIKE, top, W_SPIKE, H_SPIKE)
    
    # If all else fails just put the fucking spike somewhere random anyway
    fallback_y = random.randint(
        FLOOR_SIZE + H_SPIKE // 2 + RADIUS,
        HEIGHT - FLOOR_SIZE - H_SPIKE // 2 - RADIUS
    )
    if side == 'left':
        return pygame.Rect(0, fallback_y - H_SPIKE // 2, W_SPIKE, H_SPIKE)
    return pygame.Rect(WIDTH - W_SPIKE, fallback_y - H_SPIKE // 2, W_SPIKE, H_SPIKE)

def end(score):
    # Customised message depending on score
    if not score: # How the fuck did you manage that
        rtext = "You scored no points at all. Are you trying to lose?"
    elif not score // 5:
        rtext = f"With a score of {score}, you failed to reach level 1..."
    else:
        rtext = f"You reached level {score // 5} with a score of {score}."
        rtext += " Impressive!" if score >= 100 else ""
    
    
    
    if os.name == 'nt' and int(platform.version().split(".")[0]) > 6: # Windows Vista+ systems, we can use Task Dialogs
        from dialog import taskDialog, icons
        again = False
        result = taskDialog(
            title="Game Over",
            instruction=rtext,
            content="Would you like to play again?",
            icon=icons['question'],
            buttons=("YES", "NO")
        )
        return True if result == 6 else False
    else: # Non-Windows systems OR older than Windows Vista, just use messagebox
        from tkinter.messagebox import showinfo, INFO, YESNO
        # This boolean hack works because 
        # - YESNO returns "yes" or "no"
        # - "no"[2:] is an empty string which is False when cast to bool
        # - "yes"[2:] is "s" which is True when cast to bool
        again = bool(showinfo(
            "Game Over",
            rtext,
            detail="Would you like to play again?",
            type=YESNO,
            icon=INFO
        )[2:])
        return again

def main():
    global MAX_SPIKES
    MAX_SPIKES = 4
    
    screen = pygame.display.set_mode((WIDTH, HEIGHT))
    pygame.display.set_caption("Bounce")
    clock = pygame.time.Clock()
    font = pygame.font.SysFont('monospace', 16, bold=True)

    x, y, velocity, waiting, vx = reset_state()
    score = 0
    level = 0
    spikes = []
    
    JUMP_KEYS = (pygame.K_SPACE, pygame.K_UP, pygame.K_RETURN)

    while True:
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                pygame.quit()
                sys.exit()
            if event.type == pygame.KEYDOWN and event.key in JUMP_KEYS:
                if waiting:
                    waiting = False
                    vx = HSPEED
                velocity = JUMP_VELOCITY
                SOUNDS['jump'].play()

        if not waiting:
            velocity += GRAVITY
            y += velocity
            
            # Check if player hit the floor/roof or a spike
            death_by_floor = y + RADIUS >= (HEIGHT - FLOOR_SIZE) or y - RADIUS <= FLOOR_SIZE
            dead = death_by_floor or any(circle_rect_collides(x, y, RADIUS, spike) for spike in spikes)
            
            if dead: # Well fuck
                SOUNDS['crash'].play()
                time.sleep(0.3)
                if not end(score):
                    pygame.quit()
                    sys.exit()
                else:    
                    x, y, velocity, waiting, vx = reset_state()
                    score = 0
                    level = 0
                    MAX_SPIKES = 4
                    spikes = []
                    
            x += vx
            if x - RADIUS <= 0 or x + RADIUS >= WIDTH: # Bounce off walls
                score += 1
                new_level = score // 5
                if new_level > level:
                    level = new_level
                    vx = -vx/abs(vx) * (abs(vx) + HSPEED_DELTA) # Bounce AND speed up
                    SOUNDS['levelup'].play()
                else:
                    vx = -vx # Just bounce back normally
                    SOUNDS['bounce'].play()
                side = 'left' if vx < 0 else 'right'
                spikes.append(spawn_spike(side, spikes))
                if len(spikes) > MAX_SPIKES:
                    spikes.pop(0)
                
                # Increase number of spikes every 15 points
                if not score % 15 and score:
                    MAX_SPIKES += 1
        
        
        screen.fill(BACKGROUNDS[min(score // 5, len(BACKGROUNDS) - 1)])
        pygame.draw.rect(screen, COLORS['line'], (0, 0, WIDTH, FLOOR_SIZE))
        pygame.draw.rect(screen, COLORS['line'], (0, HEIGHT - FLOOR_SIZE, WIDTH, FLOOR_SIZE))
        pygame.draw.circle(screen, COLORS['ball'], (int(x), int(y)), RADIUS)
        pygame.draw.circle(screen, COLORS['ball_border'], (int(x), int(y)), RADIUS, 3)
        for spike in spikes:
            pygame.draw.rect(screen, COLORS['spike'], spike)
        
        score_text = font.render(str(score), True, (0, 0, 0))
        screen.blit(score_text, (WIDTH//2 - score_text.get_width()//2, 1))
        
        pygame.display.flip()
        clock.tick(FPS)


if __name__ == "__main__":

    main()
